const express = require("express");
const router = express.Router();
const db = require("../models/db");

router.get("/", async (req, res) => {
  const {
    class: classFilter,
    month,
    subjectId,
    studentId,
    session,
  } = req.query;

  // Base query to fetch the report data
  let query = `
    SELECT 
      s.id as studentId, 
      s.name, 
      s.custom_id, 
      c.class_name, 
      DATE_FORMAT(m.date, '%d-%m-%Y') as examDate, -- Format date as DD-MM-YYYY
      sub.subject_name as subject, 
      m.marks_obtained, 
      m.total_marks, 
      m.session, 
      co.course_name, 
      m.test_type, 
      m.practice_type
    FROM student s
    JOIN class c ON s.class_id = c.id
    JOIN marks m ON s.id = m.student_id
    JOIN subject sub ON m.subject_id = sub.id
    JOIN course co ON m.course_id = co.id
  `;

  // Conditions array to build WHERE clause dynamically
  const conditions = [];
  const params = [];

  // Filter by class
  if (classFilter) {
    conditions.push("c.class_name = ?");
    params.push(classFilter); // class name
  }

  // Filter by month (extracting the month from the examDate)
  if (month) {
    conditions.push("MONTH(m.date) = ?");
    params.push(month); // month number
  }

  // Filter by subject
  if (subjectId) {
    conditions.push("sub.id = ?");
    params.push(subjectId); // subject id
  }

  // Filter by studentId
  if (studentId) {
    conditions.push("s.id = ?");
    params.push(studentId); // student id
  }

  // Filter by session
  if (session) {
    conditions.push("m.session = ?");
    params.push(session); // session value
  }

  // Append WHERE clause if there are conditions
  if (conditions.length > 0) {
    query += " WHERE " + conditions.join(" AND ");
  }

  // Order by student and exam date
  query += " ORDER BY s.id, m.date;";

  try {
    const [results] = await db.query(query, params);

    // Format results into grouped student data with `onlinePractice` and `offlinePractice` flags
    const formattedResults = results.reduce((acc, row) => {
      let student = acc.find((s) => s.studentId === row.studentId);
      if (!student) {
        student = {
          studentId: row.studentId,
          name: row.name,
          custom_id: row.custom_id,
          class: row.class_name,
          report: [],
        };
        acc.push(student);
      }

      // Add `onlinePractice` and `offlinePractice` flags
      const onlinePractice =
        row.practice_type === "practice" && row.test_type === "online"
          ? "Yes"
          : "No";
      const offlinePractice =
        row.practice_type === "practice" && row.test_type === "offline"
          ? "Yes"
          : "No";

      student.report.push({
        examDate: row.examDate, // Already formatted as DD-MM-YYYY
        subject: row.subject,
        marksObtained: row.marks_obtained,
        totalMarks: row.total_marks,
        testType: row.test_type, // Return raw value directly
        practiceType: row.practice_type, // Return raw value directly
        session: row.session,
        course: row.course_name,
        onlinePractice, // New flag
        offlinePractice, // New flag
      });

      return acc;
    }, []);

    res.json(formattedResults);
  } catch (err) {
    console.error("Error fetching student reports:", err);
    res.status(500).json({ error: "Database error" });
  }
});

router.get("/performance/class", async (req, res) => {
  const query = `
    SELECT class_id, AVG(marks_obtained) AS average_score
    FROM marks
    GROUP BY class_id;
  `;
  try {
    const [results] = await db.query(query);
    res.json(results);
  } catch (err) {
    console.error("Error fetching performance data:", err.message);
    res.status(500).json({ error: "Failed to fetch performance data." });
  }
});

module.exports = router;
