const express = require("express");
const router = express.Router();
const db = require("../models/db");

router.get("/distribution", async (req, res) => {
  const query = `
    SELECT c.class_name, COUNT(s.id) AS student_count
    FROM class c
    INNER JOIN student s ON c.id = s.class_id
    GROUP BY c.class_name
    ORDER BY student_count DESC; -- Optional: Order by student count
  `;

  try {
    const [results] = await db.query(query);
    res.json(results);
  } catch (error) {
    console.error("Error fetching class distribution:", error.message);
    res.status(500).json({ error: "Failed to fetch class distribution." });
  }
});

router.get("/count/count", async (req, res) => {
  try {
    const query = "SELECT COUNT(*) AS total_classes FROM class";
    const [results] = await db.query(query);
    res.status(200).json({ total_classes: results[0].total_classes });
  } catch (err) {
    console.error("Error fetching total classes:", err.messaFge);
    res.status(500).json({ error: "Failed to fetch total classes." });
  }
});

// Create a class
router.post("/create", async (req, res) => {
  const { class_name, session } = req.body;
  const query = "INSERT INTO class (class_name, session) VALUES (?, ?)";

  try {
    const [result] = await db.query(query, [class_name, session]);
    res.status(200).json({
      message: "Class created successfully",
      classId: result.insertId,
    });
  } catch (err) {
    console.error("Database error during creation:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Get all classes
router.get("/", async (req, res) => {
  const query = "SELECT * FROM class";

  try {
    const [results] = await db.query(query);
    res.json(results);
  } catch (err) {
    console.error("Database error during fetching:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Get a single class by ID
router.get("/:id", async (req, res) => {
  const { id } = req.params;
  const query = "SELECT * FROM class WHERE id = ?";

  try {
    const [result] = await db.query(query, [id]);
    if (result.length === 0) {
      return res.status(404).json({ message: "Class not found" });
    }
    res.json(result[0]);
  } catch (err) {
    console.error("Database error during fetching single record:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Update a class by ID
router.put("/:id", async (req, res) => {
  const { id } = req.params;
  const { class_name, session } = req.body;
  const query = "UPDATE class SET class_name = ?, session = ? WHERE id = ?";

  try {
    const [result] = await db.query(query, [class_name, session, id]);
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Class not found" });
    }
    res.status(200).json({ message: "Class updated successfully" });
  } catch (err) {
    console.error("Database error during update:", err.message);
    res.status(500).json({ error: err.message });
  }
});

// Delete a class by ID with cascading dependent record deletion
router.delete("/:id", async (req, res) => {
  const { id } = req.params;
  console.log("Received request to delete class with ID:", id);

  try {
    // Step 1: Delete students directly associated with this class
    await db.query("DELETE FROM student WHERE class_id = ?", [id]);

    // Step 2: Delete marks for subjects associated with courses in this class
    await db.query(
      `DELETE m FROM marks m 
       JOIN subject sb ON m.subject_id = sb.id 
       WHERE sb.class_id = ?`,
      [id]
    );

    // Step 3: Delete marks for students associated with courses in this class
    await db.query(
      `DELETE m FROM marks m 
       JOIN student s ON m.student_id = s.id 
       WHERE s.class_id = ?`,
      [id]
    );

    // Step 4: Delete subjects and courses related to this class
    await db.query("DELETE FROM subject WHERE class_id = ?", [id]);
    await db.query("DELETE FROM course WHERE class_id = ?", [id]);

    // Step 5: Finally, delete the class itself
    const [result] = await db.query("DELETE FROM class WHERE id = ?", [id]);

    if (result.affectedRows === 0) {
      console.log("Class not found with ID:", id);
      return res.status(404).json({ error: "Class not found" });
    }

    res
      .status(200)
      .json({ message: "Class and related records deleted successfully" });
  } catch (err) {
    console.error("Database error during deletion:", err.message);
    res
      .status(500)
      .json({ error: "Failed to delete class due to foreign key constraints" });
  }
});

module.exports = router;
