const express = require("express");
const router = express.Router();
const db = require("../models/db");

// Fetch all marks entries with student name, custom ID, and subject name
router.get("/", async (req, res) => {
  const query = `
    SELECT 
      m.*, 
      s.name AS student_name, 
      s.custom_id AS student_custom_id, 
      sub.subject_name 
    FROM marks m
    JOIN student s ON m.student_id = s.id
    JOIN subject sub ON m.subject_id = sub.id
  `;
  try {
    const result = await db.query(query);
    res.status(200).json(result[0]);
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Enter marks for a student
router.post("/enter", async (req, res) => {
  const {
    student_id,
    subject_id,
    class_id,
    course_id,
    session,
    marks_obtained,
    total_marks,
    date,
    test_type,
    practice_type,
  } = req.body;

  const query = `
    INSERT INTO marks (student_id, subject_id, class_id, course_id, session, marks_obtained, total_marks, date, test_type, practice_type) 
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
  `;

  try {
    const result = await db.query(query, [
      student_id,
      subject_id,
      class_id,
      course_id,
      session,
      marks_obtained,
      total_marks,
      date,
      test_type,
      practice_type,
    ]);
    res.status(200).json({
      message: "Marks added successfully",
      marksId: result[0].insertId,
    });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Edit marks entry
router.put("/edit/:id", async (req, res) => {
  const {
    student_id,
    subject_id,
    class_id,
    course_id,
    session,
    marks_obtained,
    total_marks,
    date,
    test_type,
    practice_type,
  } = req.body;
  const { id } = req.params;

  const query = `
    UPDATE marks 
    SET student_id = ?, subject_id = ?, class_id = ?, course_id = ?, session = ?, 
        marks_obtained = ?, total_marks = ?, date = ?, test_type = ?, practice_type = ? 
    WHERE id = ?
  `;

  try {
    await db.query(query, [
      student_id,
      subject_id,
      class_id,
      course_id,
      session,
      marks_obtained,
      total_marks,
      date,
      test_type,
      practice_type,
      id,
    ]);
    res.status(200).json({ message: "Marks updated successfully" });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Delete marks entry
router.delete("/delete/:id", async (req, res) => {
  const { id } = req.params;

  const query = `DELETE FROM marks WHERE id = ?`;

  try {
    await db.query(query, [id]);
    res.status(200).json({ message: "Marks entry deleted successfully" });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Weekly report
router.get("/report/weekly", async (req, res) => {
  const { class_id, session, startDate, endDate } = req.query;
  const query = `
    SELECT 
      s.name AS student_name, 
      s.custom_id AS student_custom_id,
      sub.subject_name, 
      m.marks_obtained, 
      m.total_marks, 
      m.date, 
      m.test_type, 
      m.practice_type
    FROM marks m
    JOIN student s ON m.student_id = s.id
    JOIN subject sub ON m.subject_id = sub.id
    WHERE m.class_id = ? 
      AND m.session = ? 
      AND m.date BETWEEN ? AND ?
  `;

  try {
    const result = await db.query(query, [
      class_id,
      session,
      startDate,
      endDate,
    ]);
    res.status(200).json({ report: result[0] });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Monthly report
router.get("/report/monthly", async (req, res) => {
  const { class_id, session, month, year } = req.query;
  const query = `
    SELECT 
      s.name AS student_name, 
      s.custom_id AS student_custom_id,
      sub.subject_name, 
      m.marks_obtained, 
      m.total_marks, 
      m.date, 
      m.test_type, 
      m.practice_type
    FROM marks m
    JOIN student s ON m.student_id = s.id
    JOIN subject sub ON m.subject_id = sub.id
    WHERE m.class_id = ? 
      AND m.session = ? 
      AND MONTH(m.date) = ? 
      AND YEAR(m.date) = ?
  `;

  try {
    const result = await db.query(query, [class_id, session, month, year]);
    res.status(200).json({ report: result[0] });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

module.exports = router;
