const express = require("express");
const router = express.Router();
const db = require("../models/db");

// Create a course
router.post("/create", async (req, res) => {
  const { course_name, class_id, session } = req.body;

  if (!course_name || !class_id || !session) {
    return res.status(400).json({ error: "All fields are required" });
  }

  const query =
    "INSERT INTO course (course_name, class_id, session) VALUES (?, ?, ?)";

  try {
    const [result] = await db.query(query, [course_name, class_id, session]);
    res.status(201).json({
      message: "Course created successfully",
      courseId: result.insertId,
    });
  } catch (err) {
    console.error("Database error:", err.message);
    res.status(500).json({ error: "Failed to create course" });
  }
});

// Get all courses or filter by class_id if provided
router.get("/", async (req, res) => {
  const { class_id } = req.query;
  let query = "SELECT * FROM course";
  const params = [];

  if (class_id) {
    query += " WHERE class_id = ?";
    params.push(class_id);
  }

  try {
    const [results] = await db.query(query, params);
    res.status(200).json(results);
  } catch (err) {
    console.error("Database error:", err.message);
    res.status(500).json({ error: "Failed to fetch courses" });
  }
});

// Get a single course by ID
router.get("/:id", async (req, res) => {
  const { id } = req.params;

  if (!id) {
    return res.status(400).json({ error: "Course ID is required" });
  }

  const query = "SELECT * FROM course WHERE id = ?";

  try {
    const [result] = await db.query(query, [id]);

    if (result.length === 0) {
      return res.status(404).json({ message: "Course not found" });
    }

    res.status(200).json(result[0]);
  } catch (err) {
    console.error("Database error:", err.message);
    res.status(500).json({ error: "Failed to fetch course" });
  }
});

// Update a course by ID
router.put("/:id", async (req, res) => {
  const { id } = req.params;
  const { course_name, class_id, session } = req.body;

  if (!course_name || !class_id || !session) {
    return res.status(400).json({ error: "All fields are required" });
  }

  const query =
    "UPDATE course SET course_name = ?, class_id = ?, session = ? WHERE id = ?";

  try {
    const [result] = await db.query(query, [
      course_name,
      class_id,
      session,
      id,
    ]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ error: "Course not found" });
    }

    res.status(200).json({ message: "Course updated successfully" });
  } catch (err) {
    console.error("Database error:", err.message);
    res.status(500).json({ error: "Failed to update course" });
  }
});

// Delete a course by ID with cascading dependent record deletion
router.delete("/:id", async (req, res) => {
  const { id } = req.params;
  console.log("Received request to delete course with ID:", id);

  try {
    // Step 1: Delete marks related to this course
    await db.query("DELETE FROM marks WHERE course_id = ?", [id]);

    // Step 2: Delete students linked to this course
    await db.query("DELETE FROM student WHERE course_id = ?", [id]);

    // Step 3: Delete subjects related to this course
    await db.query("DELETE FROM subject WHERE course_id = ?", [id]);

    // Step 4: Delete the course itself
    const [result] = await db.query("DELETE FROM course WHERE id = ?", [id]);

    if (result.affectedRows === 0) {
      console.log("Course not found with ID:", id);
      return res.status(404).json({ error: "Course not found" });
    }

    res.status(200).json({ message: "Course deleted successfully" });
  } catch (err) {
    console.error("Database error:", err.message);
    res.status(500).json({ error: "Failed to delete course" });
  }
});

module.exports = router;
